/*
 * Copyright 1997-2022 Unidata Program Center/University Corporation for
 * Atmospheric Research, P.O. Box 3000, Boulder, CO 80307,
 * support@unidata.ucar.edu.
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

package ucar.unidata.data.grid;

import org.jdom2.Element;
import org.jdom2.JDOMException;
import org.jdom2.input.SAXBuilder;
import ucar.nc2.Attribute;
import ucar.nc2.constants.CDM;
import ucar.nc2.dt.GridDataset;
import ucar.nc2.dt.GridDatatype;
import ucar.nc2.grib.GribResourceReader;
import ucar.unidata.util.IOUtil;
import ucar.unidata.util.StringUtil2;
import java.io.IOException;
import java.io.InputStream;
import java.util.*;


public class GribVariableRenamer {
    private static final org.slf4j.Logger logger = org.slf4j.LoggerFactory.getLogger(GribVariableRenamer.class);

    private static HashMap<String, Renamer> map1;
    private static HashMap<String, Renamer> map2;

    //////////////////////////////////////////////// `
    // Map<String, List<String>> mapr = new HashMap<String, List<String>>();

    private void initMap1() {
        List<VariableRenamerBean> beans = readVariableRenameFile("/ucar/unidata/data/grid/grib1VarMap.xml");
        map1 = makeMapBeans(beans);
    }

    private void initMap2() {
        List<VariableRenamerBean> beans = readVariableRenameFile("/ucar/unidata/data/grid/grib2VarMap.xml");
        map2 = makeMapBeans(beans);
    }

    public List<String> getMappedNamesGrib2(String oldName) {
        if (map2 == null)
            initMap2();
        List<String> result = new ArrayList<>();
        Renamer mbean = map2.get(oldName);
        if (mbean == null)
            return null;
        for (VariableRenamerBean r : mbean.newVars) {
            result.add(r.newName);
        }
        return result;
    }

    public List<String> getMappedNamesGrib1(String oldName) {
        if (map1 == null)
            initMap1();
        List<String> result = new ArrayList<>();
        Renamer mbean = map1.get(oldName);
        if (mbean == null)
            return null;
        for (VariableRenamerBean r : mbean.newVars) {
            result.add(r.newName);
        }
        return result;
    }

    /**
     * Look for possible matches of old (4.2) grib names in new (4.3) dataset.
     *
     * @param gds check existence in this dataset. Must be from a GRIB1 or GRIB2 dataset.
     * @param oldName old name from 4.2 dataset
     * @return list of possible matches (as grid short name), each exists in the dataset
     */
    public List<String> matchNcepNames(GridDataset gds, String oldName) {
        List<String> result = new ArrayList<>();

        // look for exact match
        if (contains(gds, oldName)) {
            result.add(oldName);
            return result;
        }

        Attribute att = gds.findGlobalAttributeIgnoreCase(CDM.FILE_FORMAT);
        boolean isGrib1 = (att != null) && att.getStringValue().startsWith("GRIB-1");
        boolean isGrib2 = (att != null) && att.getStringValue().startsWith("GRIB-2");
        HashMap<String, Renamer> map;
        if (isGrib1) {
            if (map1 == null)
                initMap1();
            map = map1;

        } else if (isGrib2) {
            if (map2 == null)
                initMap2();
            map = map2;

        } else {
            return result; // empty list
        }

        // look in our renamer map
        Renamer mbean = map.get(oldName);
        if (mbean != null && mbean.newName != null && contains(gds, mbean.newName)) {
            result.add(mbean.newName); // if its unique, then we are done
            return result;
        }

        // not unique - match against NCEP dataset
        if (mbean != null) {
            String dataset = extractDatasetFromLocation(gds.getLocation());
            for (VariableRenamerBean r : mbean.newVars) {
                if (r.getDatasetType().equals(dataset) && contains(gds, r.newName))
                    result.add(r.newName);
            }
            if (result.size() == 1)
                return result; // return if unique
        }

        // not unique, no unique match against dataset - check existence in the dataset
        result.clear();
        if (mbean != null) {
            for (VariableRenamerBean r : mbean.newVarsMap.values()) {
                if (contains(gds, r.newName))
                    result.add(r.newName);
            }
            if (result.size() > 0)
                return result;
        }

        // try to map oldName -> new prefix
        result.clear();
        String oldMunged = munge(oldName);
        for (GridDatatype grid : gds.getGrids()) {
            String newMunged = munge(grid.getShortName());
            if (newMunged.startsWith(oldMunged))
                result.add(grid.getShortName());
        }
        if (result.size() > 0)
            return result;

        // return empty list
        return result;
    }

    public List<String> matchNcepNames(String datasetType, String oldName) {
        boolean isGrib1 = datasetType.endsWith("grib1");
        List<String> result = new ArrayList<>();

        HashMap<String, Renamer> map;
        if (isGrib1) {
            if (map1 == null)
                initMap1();
            map = map1;

        } else {
            if (map2 == null)
                initMap2();
            map = map2;
        }

        // look in our renamer map
        Renamer mbean = map.get(oldName);
        if (mbean != null && mbean.newName != null) {
            result.add(mbean.newName); // if its unique, then we are done
            return result;
        }

        // not unique - match against NCEP datasetType
        if (mbean != null) {
            for (VariableRenamerBean r : mbean.newVars) {
                if (r.getDatasetType().equals(datasetType))
                    result.add(r.newName);
            }
        }

        return result;
    }


    private String munge(String old) {
        StringBuilder oldLower = new StringBuilder(old.toLowerCase());
        StringUtil2.removeAll(oldLower, "_-");
        return oldLower.toString();
    }

    private boolean contains(GridDataset gds, String name) {
        return gds.findGridByShortName(name) != null;
    }

    private String getNewName(HashMap<String, Renamer> map, String datasetLocation, String oldName) {
        Renamer mbean = map.get(oldName);
        if (mbean == null)
            return null; // ??
        if (mbean.newName != null)
            return mbean.newName; // if its unique, then we are done
        String dataset = extractDatasetFromLocation(datasetLocation);
        for (VariableRenamerBean r : mbean.newVars) {
            if (r.getDatasetType().equals(dataset))
                return r.getNewName();
        }
        return null; // ??
    }

    private static String extractDatasetFromLocation(String location) {
        // check if location is from 4.5 server, and if so, remove the trailing /GC
        // This is only a temporary fix, as this renaming was only supposed to be applied
        // for the 4.2 to 4.3 transition. Thus, I plan on marking this class as deprecated
        // in the 4.5 branch
        if (location.endsWith("/GC")) {
            int locLen = location.length();
            location = location.substring(0, locLen - 3);
        }
        int pos = location.lastIndexOf("/");
        if (pos > 0)
            location = location.substring(pos + 1);
        int posSuffix = location.lastIndexOf(".");
        if (posSuffix - 14 > 0)
            return location.substring(0, posSuffix - 14) + location.substring(posSuffix);
        return "";
    }

    /*
     * <dataset name="DGEX_Alaska_12km_20100524_0000.grib2">
     * <param oldName="Geopotential_height" newName="Geopotential_height_pressure" varId="VAR_0-3-5_L100" />
     * <param oldName="Geopotential_height_surface" newName="Geopotential_height_surface" varId="VAR_0-3-5_L1" />
     * <param oldName="MSLP_Eta_Reduction" newName="MSLP_Eta_model_reduction_msl" varId="VAR_0-3-192_L101" />
     * <param oldName="Maximum_temperature" newName="Maximum_temperature_height_above_ground" varId="VAR_0-0-4_L103" />
     * <param oldName="Minimum_temperature" newName="Minimum_temperature_height_above_ground" varId="VAR_0-0-5_L103" />
     * </dataset>
     */

    // debugging only
    List<VariableRenamerBean> readVariableRenamerBeans(String which) {
        if (which.equals("GRIB-1"))
            return readVariableRenameFile("resources/grib1/grib1VarMap.xml");
        else
            return readVariableRenameFile("resources/grib2/grib2VarMap.xml");
    }

    private List<VariableRenamerBean> readVariableRenameFile(String path) {
        java.util.List<VariableRenamerBean> beans = new ArrayList<>(1000);
        try (InputStream is = IOUtil.getInputStream(path, getClass())) {
            if (is == null) {
                logger.warn("Cant read file " + path);
                return null;
            }

            SAXBuilder builder = new SAXBuilder();
            org.jdom2.Document doc = builder.build(is);
            Element root = doc.getRootElement();
            List<Element> dsElems = root.getChildren("dataset");
            for (Element dsElem : dsElems) {
                String dsName = dsElem.getAttributeValue("name");
                List<Element> params = dsElem.getChildren("param");
                for (Element elem : params) {
                    String oldName = elem.getAttributeValue("oldName");
                    String newName = elem.getAttributeValue("newName");
                    String varId = elem.getAttributeValue("varId");
                    beans.add(new VariableRenamerBean(dsName, oldName, newName, varId));
                }
            }
            return beans;

        } catch (IOException | JDOMException ioe) {
            ioe.printStackTrace();
            return null;
        }
    }

    @Deprecated
    public static class VariableRenamerBean implements Comparable<VariableRenamerBean> {
        String dsName, dsType, oldName, newName, varId;

        // no-arg constructor
        public VariableRenamerBean() {}

        public VariableRenamerBean(String dsName, String oldName, String newName, String varId) {
            this.dsName = dsName;
            this.dsType = extractDatasetFromLocation(dsName);
            this.oldName = oldName;
            this.newName = newName;
            this.varId = varId;
        }

        public String getDataset() {
            return dsName;
        }

        public String getDatasetType() {
            return dsType;
        }

        public String getVarId() {
            return varId;
        }

        public String getOldName() {
            return oldName;
        }

        public String getNewName() {
            return newName;
        }

        public String getStatus() {
            if (oldName.equals(newName))
                return "*";
            if (oldName.equalsIgnoreCase(newName))
                return "**";
            return "";
        }

        @Override
        public int compareTo(VariableRenamerBean o) {
            return newName.compareTo(o.getNewName());
        }
    }

    //////////////////////////////////////////////////

    private HashMap<String, Renamer> makeMapBeans(List<VariableRenamerBean> vbeans) {
        HashMap<String, Renamer> map = new HashMap<>(3000);
        for (VariableRenamerBean vbean : vbeans) {

            // construct the old -> new mapping
            Renamer mbean = map.get(vbean.getOldName());
            if (mbean == null) {
                mbean = new Renamer(vbean.getOldName());
                map.put(vbean.getOldName(), mbean);
            }
            mbean.add(vbean);
        }

        for (Renamer rmap : map.values()) {
            rmap.finish();
        }

        return map;
    }

    @Deprecated
    private static class Renamer {
        String oldName, newName; // newName exists when theres only one
        List<VariableRenamerBean> newVars = new ArrayList<>();
        HashMap<String, VariableRenamerBean> newVarsMap = new HashMap<>();

        // no-arg constructor
        public Renamer() {}

        public Renamer(String oldName) {
            this.oldName = oldName;
        }

        void add(VariableRenamerBean vbean) {
            newVarsMap.put(vbean.getNewName(), vbean);
            newVars.add(vbean);
        }

        void finish() {
            if (newVarsMap.values().size() == 1) {
                newName = newVars.get(0).getNewName();
            }
        }

        public int getCount() {
            return newVars.size();
        }

        public String getOldName() {
            return oldName;
        }

    }

    public static void main(String[] args) throws IOException {
        ucar.nc2.dt.grid.GridDataset gds =
                ucar.nc2.dt.grid.GridDataset.open("Q:/cdmUnitTest/tds/ncep/GFS_CONUS_80km_20100513_0600.grib1");

        GribVariableRenamer r = new GribVariableRenamer();
        List<String> result = r.matchNcepNames(gds, "Precipitable_water");
    }

}
