/*
 * Copyright (C) ITC 2007 - International Institute for Geo-Information Science and Earth Observation
 *
 * Author: Willem Nieuwenhuis 
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
*/
package org.itc.idv.math.test;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;

import junit.framework.Assert;
import junit.framework.TestCase;

import org.itc.idv.math.SunriseSunsetCollector;

public class SunriseSunsetCollectorTest extends TestCase {

	SunriseSunsetCollector collector;
	private ArrayList<Calendar> dates;
	
	public SunriseSunsetCollectorTest(String name) {
		super(name);
	}

	@Override
	protected void setUp() throws Exception {
		super.setUp();
		dates = new ArrayList<Calendar>();
		collector = new SunriseSunsetCollector(dates);
	}

	@Override
	protected void tearDown() throws Exception {
		super.tearDown();
	}

	/*
	 * Test method for 'org.n52.math.SunriseSunsetCollector.calculate(double, double)'
	 */
	public void testCalculateSingle() {
		// Test the case for a single input date
		// At 52 13'N, 6 53'E, on 11 sept 2007:
		// Sunrise = 5:11 utc; sunset = 18:07 utc
		//     Source=http://www.spacepage.be/component/option,com_waarnemen/Itemid,166/task,zonkalender/
		//     Note: Seems to ignore the minutes
		// Sunrise = 4:59 utc; sunset = 17:58 utc
		//     Source = http://aa.usno.navy.mil/
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, 2007); 
		cal.set(Calendar.MONTH, 8); // September; month is zero-based in Calendar
		cal.set(Calendar.DAY_OF_MONTH, 11);
		dates.add(cal);
		List<Calendar> result = collector.calculate(52.21883, 6.88333);
		Assert.assertEquals("Exactly one pair of sunrise / sunset expected", 2, result.size());
		int hour = result.get(0).get(Calendar.HOUR_OF_DAY);
		int min = result.get(0).get(Calendar.MINUTE);
		int hourss = result.get(1).get(Calendar.HOUR_OF_DAY);
		int minss = result.get(1).get(Calendar.MINUTE);
		Assert.assertEquals("Sunrise time should be: 5:07", 5, hour);
		Assert.assertEquals("Sunrise time should be: 5:04", 4, min);
		Assert.assertEquals("Sunset time should be: 17:54", 17, hourss);
		Assert.assertEquals("Sunset time should be: 17:54", 54, minss);
		// Calculated:
		//    Sunset = 5:04 utc; sunset = 17.54 utc
	}

	/*
	 * Test method for 'org.n52.math.SunriseSunsetCollector.calculate(double, double)'
	 */
	public void testCalculateRange() {
		// Test the case for a exactly two dates; creating a range
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, 2007); 
		cal.set(Calendar.MONTH, 8); 
		cal.set(Calendar.DAY_OF_MONTH, 11);
		dates.add(cal);
		Calendar cal2 = Calendar.getInstance();
		cal2.set(Calendar.YEAR, 2007); 
		cal2.set(Calendar.MONTH, 8); 
		cal2.set(Calendar.DAY_OF_MONTH, 20);
		dates.add(cal2);
		List<Calendar> result = collector.calculate(52, 6);
		Assert.assertEquals("Ten sunrise / sunset pairs expected", 20, result.size());
	}
	/*
	 * Test method for 'org.n52.math.SunriseSunsetCollector.calculate(double, double)'
	 */
	public void testCalculateList() {
		// Test the case for a single input date
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, 2007); 
		cal.set(Calendar.MONTH, 8); 
		cal.set(Calendar.DAY_OF_MONTH, 11);
		dates.add(cal);
		Calendar cal2 = Calendar.getInstance();
		cal2.set(Calendar.YEAR, 2007); 
		cal2.set(Calendar.MONTH, 8); 
		cal2.set(Calendar.DAY_OF_MONTH, 13);
		dates.add(cal2);
		Calendar cal3 = Calendar.getInstance();
		cal3.set(Calendar.YEAR, 2007); 
		cal3.set(Calendar.MONTH, 8); 
		cal3.set(Calendar.DAY_OF_MONTH, 17);
		dates.add(cal3);
		Calendar cal4 = Calendar.getInstance();
		cal4.set(Calendar.YEAR, 2007); 
		cal4.set(Calendar.MONTH, 8); 
		cal4.set(Calendar.DAY_OF_MONTH, 19);
		dates.add(cal4);
		List<Calendar> result = collector.calculate(52, 6);
		Assert.assertEquals("Four pair of sunrise / sunset pairs expected", 8, result.size());
	}
	public void testJeffDates() {
		Calendar cal = Calendar.getInstance();
		cal.set(Calendar.YEAR, 2007); 
		cal.set(Calendar.MONTH, 9); 
		cal.set(Calendar.DAY_OF_MONTH, 17);
		dates.add(cal);
		Calendar cal2 = Calendar.getInstance();
		cal2.set(Calendar.YEAR, 2007); 
		cal2.set(Calendar.MONTH, 9); 
		cal2.set(Calendar.DAY_OF_MONTH, 18);
		dates.add(cal2);
		List<Calendar> result = collector.calculate(40, -107);
		Assert.assertEquals("Two pair of sunrise / sunset pairs expected", 4, result.size());
		int hour = result.get(0).get(Calendar.HOUR_OF_DAY);
		int min = result.get(0).get(Calendar.MINUTE);
		int hourss = result.get(1).get(Calendar.HOUR_OF_DAY);
		int minss = result.get(1).get(Calendar.MINUTE);
		Assert.assertEquals("Sunrise hour", 0, hour);
		Assert.assertEquals("Sunset hour", 13, hourss);
		Assert.assertEquals("Sunrise minutes", 16, min);
		Assert.assertEquals("Sunset minutes", 15, minss);
	}
}
