/* 
 * Copyright (C) ITC 2005 - International Institute for Geo-Information Science and Earth Observation
 * 
 * Author: Jan Hendrikse 
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
*/
package org.itc.idv.math;
import static java.lang.Math.*;
import Jama.Matrix;

@SuppressWarnings("serial")
public class Vector3D extends ColumnVector {
    static private double epsilon = 1e-10;
    static public void setEpsilon(double eps)
    {
        epsilon = eps;
    }

    public Vector3D(Matrix mat) {
        super(3);
        for (int i = 0; i < 3; i++) {
            set(i, mat.get(i, 0));
        }
    }
    
    public Vector3D(double[] V) {
        super(V, 3);
    }
    
    public Vector3D(double a, double b, double c) {
        super(3);
        set(0, a);
        set(1, b);
        set(2, c);
    }
    
    public Vector3D() {
        super(3);
    }
    
    @Override
	public boolean equals(Object obj) {
        if (obj instanceof Vector3D) {
            Vector3D vec = (Vector3D) obj;
            if (abs(get(0) - vec.get(0)) + 
                abs(get(1) - vec.get(1)) +
                abs(get(2) - vec.get(2)) 
                    < epsilon) {
                return true;
            }
        }
        return false;
    }
    
    @Override
	public String toString() {
        return get(0) + "," + get(1) + "," + get(2);
    }
    
    public Vector3D plus(Vector3D v) {
        return new Vector3D(super.plus(v));
    }
    
    public Vector3D minus(Vector3D v) {
        return new Vector3D (super.minus(v));
    }
    
    @Override
	public Vector3D times(double s) {
        return new Vector3D (super.times(s));
    }
    
    /**
     * outer product (cross-product)result = Vector3D X vec2
     * @param vec2 right-hand input vector
     * @return the outer product
     */
    public Vector3D crossProduct(ColumnVector vec2) {
    	// outer product, 'cross-product' 
        Vector3D result = new Vector3D();    
        result.set(0, get(1)*vec2.get(2) - get(2)*vec2.get(1));
        result.set(1, get(2)*vec2.get(0) - get(0)*vec2.get(2));
        result.set(2, get(0)*vec2.get(1) - get(1)*vec2.get(0));
        return result;          
    }
    
    
}
