/* 
 * Copyright (C) ITC 2005 - International Institute for Geo-Information Science and Earth Observation
 * 
 * Author: Jan Hendrikse 
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
*/
package org.itc.idv.math;

import static java.lang.Math.PI;
import static java.lang.Math.cos;
import static java.lang.Math.sin;
import static java.lang.Math.sqrt;


public class TerrainTriangulation {
    private double lat = Double.NaN;
    private double lon = Double.NaN;
    private double phi;
    private double lam;
    private double sinphi;
    private double cosphi;
    private double sinlam;
    private double coslam;
    private final double a;
    private final double f;
    public TerrainTriangulation (double a, double fInv) {
        this.a = a;
        this.f = 1/fInv;
    }
    
    public double getEquatorRadius() {
        return a;
    }
    
    public double getPhi() {
        return phi;
    }
    
    /**
     * @return lam (in radians) needed by SunTriangulation::getTrueSolarTime
     */
    public double getLam() { // 
        return lam;
    }

    /**
     * @param lat
     * @param lon
     * checks whether latlon is new and set the mebers lat and lon
     * converts latlon to radians and sets phi and lam and their 
     * sine and cosine values (all members)
     */
    private void setLatLon(double lat, double lon) {
        if ((this.lat != lat ) || (this.lon != lon)) {
            this.lat = lat;
            this.lon = lon;
            this.phi = lat * PI / 180;
            this.lam = lon * PI / 180;
            this.sinphi = sin(phi);
            this.cosphi = cos(phi);
            this.sinlam = sin(lam);
            this.coslam = cos(lam);
        }
        return;
    }
    
    /**
     * @param lat
     * @param lon
     * @return vector of unit length defined by local terrain latlon
     * and pointing upward, perpendicular to local ellipsoid surface 
     * thus being at right angles to local parallel and to local meridian
     * Needed for zenith angle calculation
     */
    public Vector3D getUnitVerticalVector(double lat, double lon) {
        setLatLon(lat, lon);
        Vector3D u = new Vector3D();
        u.set(0, coslam * cosphi);
        u.set(1, sinlam * cosphi);
        u.set(2, sinphi); 
        return u;
    }
    
    /**
     * @param lat
     * @param lon
     * @return vector of unit length defined by local terrain latlon
     * and pointing northward, tangent to local ellipsoid surface and
     * tangent to local meridian,
     * thus being at right angles to local parallel and to local vertical'
     * Needed for azimuth calculation
     */
    public Vector3D getUnitNorthVector(double lat, double lon) {
        setLatLon(lat, lon);
        Vector3D u = new Vector3D();
        u.set(0, -sinphi * coslam);
        u.set(1, -sinphi * sinlam);
        u.set(2, cosphi); 
        return u;
    }
    
    /**
     * @param lat
     * @param lon
     * @return vector of unit length defined by local terrain latlon
     * and pointing eastward, tangent to local ellipsoid surface
     * thus being at right angles to local meridian and to local vertical.
     * Needed for azimuth calculation
     */
    public Vector3D getUnitEastVector(double lat, double lon) {
        setLatLon(lat, lon);
        Vector3D u = new Vector3D();
        u.set(0, -sinlam);
        u.set(1, coslam);
        u.set(2, 0.0); 
        return u;
    }
    
    /**
     * @param lat latitude in degrees at terrain location
     * @param lon longitude in degrees at terrain location
     * @return vector from Earth center to terrain position on the 
     * ellipsoid surface and defined by ellipsoidal lat and lon
     */
    public Vector3D getTerrainPositionVector(double lat, double lon) {
        setLatLon(lat, lon);
        Vector3D vec = new Vector3D();
        double e2 = 2*f - f*f;
        /// radius of curvature in prime vertic plane:
        double N = a / sqrt(1 - e2 * sinphi * sinphi);
        vec.set(0, N * coslam * cosphi);
        vec.set(1, N * sinlam * cosphi);
        vec.set(2, N * (1 - e2)* sinphi);
        return vec;
    }
}
