/*
 * Copyright (C) ITC 2007 - International Institute for Geo-Information Science and Earth Observation
 *
 * Author: Willem Nieuwenhuis 
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
*/
package org.itc.idv.math;

import java.util.ArrayList;
import java.util.Calendar;
import java.util.List;
import java.util.TimeZone;

/**
 * Helper class for calculating sunsrise/sunset pairs.
 * The sunset/sunrise times are calculated geometrically; the influence of the
 * atmosphere as well as the diameter of the sun are not taken into account.<p>
 * The times calculated are in utc and converted to Calendar objects; the
 * Calendar objects use "GMT+0" as timezone, no daylight savings.<p>
 * 
 * @author Willem Nieuwenhuis
 */
public class SunriseSunsetCollector  {

	private List<Calendar> dates;
    private final double EQRADIUSWGS84 = 6378137.0;
    private final double INVFWGS84 = 298.257223563;

	/**
	 * Initialize the SunriseSunsetCollector with a list of dates to calculate the
	 * sunsrise/sunset pairs for.<p>
	 * 
	 * @param dates The initial dates; The only information extracted from the
	 *  Calendar objects is the d-m-y fields, time and timezone are not considered 
	 */
	public SunriseSunsetCollector(List<Calendar> dates) {
		this.dates = dates;
	}
	/**
	 * Calculate sunrise and sunset times at the location (Lat, lon)<p>
	 * The calculation of the sunrise/sunset pairs is done based on the following logic:
	 * if the list contains 1 date, or more than 2 dates then for each date the pair is
	 * calculated. If the list contains 2 dates, they are considered the begin and end date
	 * of a range and sunrise/sunset pairs are calculated for both dates as well as for all
	 * dates in between.
	 *
	 * @return a list of sunrise/sunset time pairs
	 */
	public List<Calendar> calculate(double lat, double lon) {
		ArrayList<Calendar> list = new ArrayList<Calendar>();
		if (dates.size() == 2) {
			dates = expandDates(dates); 
		}
		
		TerrainTriangulation terra = new TerrainTriangulation(EQRADIUSWGS84, INVFWGS84);
		double[] latlon = new double[2];
		latlon[0] = lat;
		latlon[1] = lon;
		for (Calendar cal : dates) {
			int year = cal.get(Calendar.YEAR);
			int month = cal.get(Calendar.MONTH);
			int day = cal.get(Calendar.DAY_OF_MONTH);
			// use noon as starting time for all dates
			SunTriangulation st = new SunTriangulation(terra, year, month + 1, day, 12);
			double sunriseUTC = st.getSunRiseAlgorithm().calculate(latlon);
			double sunsetUTC = st.getSunSetAlgorithm().calculate(latlon);
			Calendar dateSunrise = dateFromFields(year, month, day, sunriseUTC);
			Calendar dateSunset = dateFromFields(year, month, day, sunsetUTC);
			list.add(dateSunrise);
			list.add(dateSunset);
		}
		return list;
	}
	/*
	 * list contains exactly two dates
	 * Calculate all dates between and including the begin date and the end date
	 */
	private ArrayList<Calendar> expandDates(List<Calendar> list) {
		ArrayList<Calendar> expanded = new ArrayList<Calendar>();
		expanded.add(list.get(0));
		long startMillis = list.get(0).getTimeInMillis();
		long endMillis = list.get(1).getTimeInMillis();
		Calendar cal = Calendar.getInstance(TimeZone.getTimeZone("GMT+0"));
		cal.setTimeInMillis(startMillis);
		cal.add(Calendar.DAY_OF_YEAR, 1); // automatically rolls other fields
		while (cal.getTimeInMillis() <= endMillis) {
			expanded.add((Calendar) cal.clone()); // make sure to make a copy
			cal.add(Calendar.DAY_OF_YEAR, 1); // automatically rolls other fields
		}
		return expanded;
	}
	/**
	 *
	 * @param year
	 * @param month
	 * @param day
	 * @param utc
	 * @param dateSunrise
	 */
	private Calendar dateFromFields(int year, int month, int day, double utc) {
		Calendar date = Calendar.getInstance(TimeZone.getTimeZone("GMT+0"));
		date.set(Calendar.YEAR, year); 
		date.set(Calendar.MONTH, month); 
		date.set(Calendar.DAY_OF_MONTH, day);
		int h = (int) utc;
		double rest = 60 * (utc - h);
		int m = (int) rest;
		rest = 60 * (rest - m);
		int s = (int) rest;
		int ms = (int) (1000 * (rest - s));
		date.set(Calendar.HOUR_OF_DAY, h);
		date.set(Calendar.MINUTE, m);
		date.set(Calendar.SECOND, s);
		date.set(Calendar.MILLISECOND, ms);
		
		return date;
	}

}
