/* 
 * Copyright (C) ITC 2005 - International Institute for Geo-Information Science and Earth Observation
 * 
 * Author: Jan Hendrikse 
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
*/
package org.itc.idv.math;

import static java.lang.Math.*;
import numericalMethods.calculus.rootFinding.NewtonRaphson;
import numericalMethods.calculus.rootFinding.NewtonRaphson.*;
import java.util.Calendar;
import java.util.GregorianCalendar;

/**
 * Supplier class azimuth, zenith angle, distance, equation of time and
 * declination algorithms, to calculate with sun triangulation. SunTriangulation
 * holds the geometric properties (3D) of the triangle ETS: <table border="0"
 * cellspacing="3">
 * <tr>
 * <td>E
 * <td> earth-center = (0.0.0),
 * <tr>
 * <td>T
 * <td> observed terrain-pixel position,
 * <tr>
 * <td>S
 * <td> sun position; </table> All positions are metric, cartesian and
 * geo-centric. T depends on assumptions about ellipsoid and image georeference.
 * terrTriangulation contains the geometry depending merely on T. S depends for
 * the moment on 3 time parameters: year, month, dayInMonth. Its coords are
 * expressed in an earth-fixed, earth-centered, cartesian system The purpose is
 * to produce (output) parameters for (a.o) solar reflection:
 * <ol>
 * <li>{@linkplain #getZenithAngleAlgorithm zenith angle}, angle between local
 * vertical and sun direction;
 * <li>{@linkplain #getAzimuthAlgorithm azimuth}, angle between local North
 * and ground-projected sun direction;
 * <li>{@linkplain #getDistanceAlgorithm distance}, distance from terrain
 * point to sun (meters);
 * <li>{@linkplain #getSunRiseAlgorithm sun rise}, moment of sun rise in utc
 * hours;
 * <li>{@linkplain #getSunSetAlgorithm sun set}, moment of sun set in utc
 * hours;
 * <li>{@linkplain #getDayLightAlgorithm daylight}, daylight duration in hours;
 * <li>{@linkplain #getSolarNoonAlgorithm solar noon}, moment of solar noon in
 * utc hours;
 * <li>{@linkplain #getTrueSolarNoonAlgorithm true solar noon}, moment of solar noon in
 * utc hours, more accurate than the previous algorithm;
 * </ol>
 * 
 * @author Jan Hendrikse
 */
public class SunTriangulation{
    /*
     * private double getSunHeight(double utc, double lat, double lon) { int
     * year = calendar.get(Calendar.YEAR); int month =
     * calendar.get(Calendar.MONTH) + 1; int day =
     * calendar.get(Calendar.DAY_OF_MONTH); SunTriangulation st = new
     * SunTriangulation(terrTriangulation, year, month, day, utc);
     * 
     * return PI - st.getZenithAngle(lat, lon); }
     */
    
    private class NewtonSunHeight implements RealFunctionWithDerivative {
        private double lat;
    
        private double lon;
    
        public NewtonSunHeight(double lat, double lon) {
            this.lat = lat;
            this.lon = lon;
        }
    
        /**
         * returns the SunHeight given a utc time
         */
        public void eval(double utc, double[] f, int offsetF, double[] df,
                int offsetDF) {
            int year = calendar.get(Calendar.YEAR);
            int month = calendar.get(Calendar.MONTH) + 1;
            int day = calendar.get(Calendar.DAY_OF_MONTH);
            SunTriangulation st1 = new SunTriangulation(terrTriangulation,
                    year, month, day, utc);
    
            f[offsetF] = 90.0 - st1.getZenithAngle(lat, lon);
    
            // now the derivative
    
            SunTriangulation st2 = new SunTriangulation(terrTriangulation,
                    year, month, day, utc + 0.5);
    
            double f1 = 90.0 - st2.getZenithAngle(lat, lon);
    
            df[offsetDF] = (f1 - f[offsetF]) / 0.5;
        }
    }

    private class NewtonSunHeightChange implements RealFunctionWithDerivative {
        private double lat;
    
        private double lon;
    
        public NewtonSunHeightChange(double lat, double lon) {
            this.lat = lat;
            this.lon = lon;
        }
    
        /**
         * returns the SunHeightChange given a utc time
         */
        public void eval(double utc, double[] f, int offsetF, double[] df,
                int offsetDF) {
            int year = calendar.get(Calendar.YEAR);
            int month = calendar.get(Calendar.MONTH) + 1;
            int day = calendar.get(Calendar.DAY_OF_MONTH);
            SunTriangulation st1 = new SunTriangulation(terrTriangulation,
                    year, month, day, utc);
    
            f[offsetF] = st1.getElevationAngleChange(lat, lon);
    
            // now the derivative
    
            SunTriangulation st2 = new SunTriangulation(terrTriangulation,
                    year, month, day, utc + 0.5);
    
            double f1 = st2.getElevationAngleChange(lat, lon);
    
            df[offsetDF] = (f1 - f[offsetF]) / 0.5;
        }
    }
    
    private final static String[] paramDescr = { "latitude in degrees at terrain location", "longitude in degrees at terrain location" };
    private final static String[] paramNames = { "lat", "lon" };
    final double aSun = 149.59789 * 10e9;
    private double azimuth;
    private boolean azimuthCalculated;
    private GregorianCalendar calendar;
    private double distance;
    private boolean distanceCalculated;
    final double eSun = 0.017;
    private int julianday;
    private double lat = Double.NaN;
    private double lon = Double.NaN;
    private TerrainTriangulation terrTriangulation;
    private final double utc;
    private final double yearFraction;
    private double zenithAngle;
    private boolean zenithAngleCalculated;

    /**
     * returns the ZenithAngle Algorithm; it produces the zenith angle, angle
     * (in degrees) between local vertical and sun direction, given local lat
     * and lon (in degrees)
     * 
     * @return the Algorithm
     */
    public Algorithm getZenithAngleAlgorithm() {
        return new AbstractAlgorithm("Zenith Angle",
                "angle between local vertical and sun direction", paramNames,
                paramDescr) {
            public double calculate(double[] params) {
                return getZenithAngle(params[0], params[1]);
            }
        };
    }

    private void check(double lat, double lon) {
        if ((this.lat != lat) || (this.lon != lon)) {
            clear();
            this.lat = lat;
            this.lon = lon;
        }
        return;
    }

    private void clear() {
        zenithAngleCalculated = false;
        azimuthCalculated = false;
        distanceCalculated = false;
    }

    private double getAzimuth(double lat, double lon) {
        check(lat, lon);
        if (!azimuthCalculated) {
            Vector3D unitNorth = terrTriangulation.getUnitNorthVector(lat, lon);
            Vector3D unitEast = terrTriangulation.getUnitEastVector(lat, lon);
            Vector3D posVecTerr = terrTriangulation.getTerrainPositionVector(
                    lat, lon);
            Vector3D posVecSat = getSunPositionVector();
            Vector3D V2 = getTerrainToSunVector(posVecTerr, posVecSat);
            Vector3D unitV2 = V2.times(1 / V2.normF()); // reduce to unit length
    
            // /direction-cosines of PS vector with local east- and north-axes:
            double nort = unitNorth.dotProduct(unitV2);
            double east = unitEast.dotProduct(unitV2);
            azimuth = atan2(east, nort) * 180.0 / PI;
            if (azimuth < 0)
                azimuth += 360;
            azimuthCalculated = true;
        }
        return azimuth;
    }

    /**
     * returns the Azimuth Algorithm; it produces the azimuth, angle (in
     * degrees) between local North and projected sun direction, given local lat
     * and lon (in degrees)
     * 
     * @return the Algorithm
     */
    public Algorithm getAzimuthAlgorithm() {
        return new AbstractAlgorithm("Azimuth",
                "angle betw. local North and projected sun direction",
                paramNames, paramDescr) {
            public double calculate(double[] params) {
                return getAzimuth(params[0], params[1]);
            }
        };
    }

    /**
     * returns the Day Light Algorithm; it produces the duration of daylight, in
     * hours
     * 
     * @return the Algorithm
     */
    public Algorithm getDayLightAlgorithm() {
        return new AbstractAlgorithm("DayLight",
                "duration of daylight in hours", paramNames, paramDescr) {
            public double calculate(double[] params) {
                return (getSunSetTime(params[0], params[1]) - getSunRiseTime(
                        params[0], params[1]));
            }
        };
    }

    private double getDistance(double lat, double lon) {
        check(lat, lon);
        if (!distanceCalculated) {
            Vector3D posVecTerr = terrTriangulation.getTerrainPositionVector(
                    lat, lon);
            Vector3D posVecSat = getSunPositionVector();
            Vector3D V2 = getTerrainToSunVector(posVecTerr, posVecSat);
            distanceCalculated = true;
            distance = V2.length();

            distance = getSunDistance(julianday);
        }
        return distance;
    }

    /**
     * returns the Distance Algorithm; it produces the distance, between local
     * position and sun, in AU (astronomic units), given local lat and lon (in
     * degrees). 1 AU = mean distance earth - sun
     * 
     * @return the Algorithm
     */
    public Algorithm getDistanceAlgorithm() {
        return new AbstractAlgorithm("Distance",
                "distance from terrain point to sun (in AU)", paramNames,
                paramDescr) {
            public double calculate(double[] params) {
                return getDistance(params[0], params[1]);
            }
        };
    }

    private double getEquationOfTimeResult(double lat, double lon) {
        return getMinutesShiftFromEquationOfTime(yearFraction);
    }

    public Algorithm getEquationOfTimeResultAlgorithm() {
        return new AbstractAlgorithm("EqOfTime",
                "deviation in minutes from regular solar position", paramNames,
                paramDescr) {
            public double calculate(double[] params) {
                return getEquationOfTimeResult(params[0], params[1]);
            }
        };
    }

    private double getHourAngle(double trueSolarTime) {
        return (trueSolarTime / 4 - 180) * PI / 180; // in radians
    }

    private double getMinutesShiftFromEquationOfTime(double gamma) {
        final double EQT1 = 229.18;
        final double EQT2 = 0.000075;
        final double EQT3 = 0.001868;
        final double EQT4 = 0.032077;
        final double EQT5 = 0.014615;
        final double EQT6 = 0.040849;
        double EquTime = EQT1 * (EQT2 + EQT3 * cos(gamma) - EQT4 * sin(gamma));
        double EquT2 = EQT1
                * (-EQT5 * cos(2. * gamma) - EQT6 * sin(2. * gamma));
        EquTime += EquT2;
        return EquTime;
    }

    /**
     * returns the Local Noon Algorithm; it produces the solar noon, in utc
     * hours
     * 
     * @return the Algorithm
     */
    public Algorithm getSolarNoonAlgorithm() {
        return new AbstractAlgorithm("Local Noon", "solar noon in utc",
                paramNames, paramDescr) {
            public double calculate(double[] params) {
                return (getSunSetTime(params[0], params[1]) + getSunRiseTime(
                        params[0], params[1])) / 2;
            }
        };
    }

    private double getSunDeclination(double gamma) {
        final double DECL1 = 0.006918;
        final double DECL2 = 0.399912;
        final double DECL3 = 0.070257;
        final double DECL4 = 0.006758;
        final double DECL5 = 0.000907;
        final double DECL6 = 0.002697;
        final double DECL7 = 0.00148;
        double decli = DECL1 - DECL2 * cos(gamma) + DECL3 * sin(gamma) - DECL4
                * cos(2 * gamma) + DECL5 * sin(2 * gamma) - DECL6
                * cos(3 * gamma) + DECL7 * sin(3 * gamma);
        return decli;
    }

    /**
     * returns the Sun Declination Algorithm; it produces the declination, angle
     * (in degrees) between equator plane and sun direction, given utc time
     * 
     * @return the Algorithm
     */
    public Algorithm getSunDeclinationAlgorithm() {
        return new AbstractAlgorithm("Declination",
                "declination of sun with r. to earth equator plane",
                paramNames, paramDescr) {
            public double calculate(double[] params) {
                return getSunDeclinDegrees(params[0], params[1]);
            }
        };
    }

    private double getSunDeclinDegrees(double lat, double lon) {
        return getSunDeclination(yearFraction) * 180.0 / PI;
    }

    /**
     * @return distance from EarthCenter to Sun center defined by SunCentric
     *         polar coordinates: r and theta The yearly elliptic orbit of the
     *         earth around the sun is parametrized with theta, here a function
     *         of the integer julianday and assumed linear. If theta would be a
     *         linear function of time it would be violating Kepler's Second Law
     *         http://scienceworld.wolfram.com/physics/KeplersSecondLaw.html E0
     *         is the inverse of the square of the sun-distance ratio E0 =
     *         1.000110 + 0.034221*cos(theta)+0.00128*sin(theta) +
     *         0.000719*cos(2*theta)+0.000077*sin(2*theta); The coefficients of
     *         this equation must take the anomaly of the sun movement
     *         ("Equation of time") into account The calculated distance is
     *         expressed as a factor of the "average sun-distance" (on 1 Jan
     *         approx. 0.98, on 1 Jul approx. 1.01)
     *         http://curious.astro.cornell.edu/legal.php
     */
    private double getSunDistance(double julianday) {
        double theta = 2 * PI * (julianday - 1) / 365.25;
        double E0 = 1.000110 + 0.034221 * cos(theta) + 0.00128 * sin(theta)
                + 0.000719 * cos(2 * theta) + 0.000077 * sin(2 * theta);

        return 1 / E0;
    }

    /**
     * @return vector from EarthCenter to Sun center defined by GeoCentric polar
     *         coordinates: posRadius (length of vec) obtained from
     *         getSunDistance() sunLam (angle between vec and prime meridian
     *         plane) sunPsi (angle between vec and equator plane), i.e.
     *         geocentric, not geodetic (phi) latitude
     */
    private Vector3D getSunPositionVector() {
        double eqt = getMinutesShiftFromEquationOfTime(yearFraction);
        double lamLocal = terrTriangulation.getLam();
        double trueSolarT = getTrueSolarTime(lamLocal, utc, eqt);
        double sunLam = -getHourAngle(trueSolarT);
        double sunPsi = getSunDeclination(yearFraction);
        Vector3D vec = new Vector3D();
        double posRadius = getSunDistance(yearFraction / 24);
        posRadius *= aSun;
        vec.set(0, posRadius * cos(sunLam) * cos(sunPsi));
        vec.set(1, posRadius * sin(sunLam) * cos(sunPsi));
        vec.set(2, posRadius * sin(sunPsi));
        return vec;
    }

    /**
     * returns the Sun Rise Algorithm; it produces the time of sunrise, in utc
     * hours
     * 
     * @return the Algorithm
     */
    public Algorithm getSunRiseAlgorithm() {
        return new AbstractAlgorithm("SunRise", "time of sunrise in utc",
                paramNames, paramDescr) {
            public double calculate(double[] params) {
                return getSunRiseTime(params[0], params[1]);
            }
        };
    }

    private double getSunRiseTime(double lat, double lon) {
        NewtonSunHeight nsh = new NewtonSunHeight(lat, lon);
        double[] rootValues = new double[3];

        double firstGuess = 6 - abs(lon) / 15.0; // local time wrt utc
        if (lon < 0) {
        	firstGuess *= -1;
        }
        if (abs(lat) > 76)
            return Double.NaN;// inside polar circle
        NewtonRaphson.search(nsh, firstGuess, rootValues, NewtonRaphson.ACC,
                NewtonRaphson.ACC, -24, 24, 62, true);

        return rootValues[0];
    }

    /**
     * returns the True Solar Noon Algorithm; it produces the time of noon, in utc
     * hours. This Algorithm uses Newton-Raphson approximation internally, and therefore
     * gives a more accurate result than getSolarNoonAlgorithm, which simply
     * uses (sunrise + sunset) / 2.
     * 
     * @return the Algorithm
     */
    public Algorithm getTrueSolarNoonAlgorithm() {
        return new AbstractAlgorithm("True Local Noon", "True solar noon in utc",
                paramNames, paramDescr) {
            public double calculate(double[] params) {
                return (getTrueSunNoonTime(params[0], params[1]));
            }
        };
    }

    private double getTrueSunNoonTime(double lat, double lon) {
        NewtonSunHeightChange nshc = new NewtonSunHeightChange(lat, lon);
        double[] rootValues = new double[3];

        double firstGuess = 12 - abs(lon) / 15.0; // local time wrt utc
        if (lon < 0) {
        	firstGuess *= -1;
        }
        if (abs(lat) > 76)
            return Double.NaN;// inside polar circle
        NewtonRaphson.search(nshc, firstGuess, rootValues, NewtonRaphson.ACC,
                NewtonRaphson.ACC, -24, 24, 62, true);

        return rootValues[0];
    }
    
    /**
     * returns the Sun Set Algorithm; it produces the time of sunset, in utc
     * hours
     * 
     * @return the Algorithm
     */
    public Algorithm getSunSetAlgorithm() {
        return new AbstractAlgorithm("SunSet", "time of sunset in utc",
                paramNames, paramDescr) {
            public double calculate(double[] params) {
                return getSunSetTime(params[0], params[1]);
            }
        };
    }

    private double getSunSetTime(double lat, double lon) {
        NewtonSunHeight nsh = new NewtonSunHeight(lat, lon);
        double[] rootValues = new double[3];

        double firstGuess = 18 - abs(lon) / 15.0; // local time wrt utc
        if (lon < 0) {
        	firstGuess *= -1;
        }
        if (abs(lat) > 76)
            return Double.NaN;// inside polar circle
        NewtonRaphson.search(nsh, firstGuess, rootValues, NewtonRaphson.ACC,
                NewtonRaphson.ACC, -24, 24, 62, true);

        return rootValues[0];
    }

    private Vector3D getTerrainToSunVector(Vector3D vecTerrainPosition,
            Vector3D vecSunPosition) {
        return vecSunPosition.minus(vecTerrainPosition);
    }

    private double getTrueSolarTime(double lam, double utc, double eqtime_shift) {
        double time_offset = eqtime_shift - 4 * lam; // in minutes
        return utc * 60 + time_offset; // true solar time in minutes
    }

    /**
     * YearFraction is the 'fractional year' in radians from <a
     * href="www.srrb.noaa.gov/highlights/sunrise/solareqns.PDF">solareqns.pdf</a> ;
     * NOAA 2003 however modified slightly to account for leap years
     * 
     * @param year
     * @param julianday
     * @param utc
     * @return the fraction of the year of the specified date with respect to
     * januari 1st in radians
     */
    private double getYearFraction(int year, int julianday, double utc) {
        int daysInYear = 365;
        if (calendar.isLeapYear(year))
            daysInYear++;
        return 2 * Math.PI * (julianday - 1 + (utc - 12) / 24.) / daysInYear;
    }

    private double getZenithAngle(double lat, double lon) {
        check(lat, lon);
        if (!zenithAngleCalculated) {
            Vector3D unitV1 = terrTriangulation.getUnitVerticalVector(lat, lon);
            Vector3D posVecTerr = terrTriangulation.getTerrainPositionVector(
                    lat, lon);

            Vector3D posVecSun = getSunPositionVector();
            Vector3D V2 = getTerrainToSunVector(posVecTerr, posVecSun);
            Vector3D unitV2 = V2.times(1 / V2.normF()); // reduce to unit length
            zenithAngleCalculated = true;
            zenithAngle = acos(unitV1.dotProduct(unitV2)) * 180.0 / PI;
        }
        return zenithAngle;
    }
    private double getElevationAngle(double lat, double lon) {
        check(lat, lon);
        double elevationAngle = 90.0 - getZenithAngle(lat, lon);
        return elevationAngle;
    }
    private double getElevationAngleChange(double lat, double lon) {
        check(lat, lon);
        double delta = 0.001; // for numerical derivative of elevationAngle
        double elevationAngle1 = 90.0 - getZenithAngle(lat, lon);
        int year = calendar.get(Calendar.YEAR);
        int month = calendar.get(Calendar.MONTH) + 1;
        int day = calendar.get(Calendar.DAY_OF_MONTH);
        SunTriangulation timePlusDelta = new SunTriangulation( terrTriangulation, year, month, day, utc + delta);
        Algorithm zenAngleAlg = timePlusDelta.getZenithAngleAlgorithm();
        double [] params = new double [2];
        params[0] = lat;
        params[1] = lon;
        double elevationAngle2 = 90.0 - zenAngleAlg.calculate(params);
        
        double elevationAngleChange = (elevationAngle2 - elevationAngle1) / delta;
        
        return elevationAngleChange;
    }
    
    /**
     * returns the ElevationAngle Algorithm; it produces the elevation angle, 
     * angle (in degrees) between local horizon plane and sun direction, 
     * given local lat and lon (in degrees)
     * It is the complementary angle of the sun zenith angle
     * @return the Algorithm
     */
    public Algorithm getElevationAngleAlgorithm() {
        return new AbstractAlgorithm("Elevation Angle",
                "angle between local horizon-plane and sun direction", paramNames,
                paramDescr) {
            public double calculate(double[] params) {
                return getElevationAngle(params[0], params[1]);
            }
        };
    }
    /**
     * constructor of SunTriangulation using sun- and satellite -related
     * geometric input to allow mainly angle computations. The time parameters
     * yield the members int julianday (DOY) and the angle of the sun position
     * since julianday 1, 12 a.m. as input 'gamma' for Equation of time and
     * sun-Declination
     * 
     * @param terrTri
     *            holds terrainlocation specific vectors
     * @param year
     * @param month (1-12)
     * @param dayInMonth (1-31)
     * @param utc
     *            Universal Time Crd in double precision
     */
    public SunTriangulation(TerrainTriangulation terrTri, int year, int month,
            int dayInMonth, double utc) {
        this.calendar = new GregorianCalendar(year, month - 1, dayInMonth);
        this.julianday = calendar.get(Calendar.DAY_OF_YEAR);
        this.terrTriangulation = terrTri;
        this.utc = utc;
        this.yearFraction = getYearFraction(year, julianday, utc);
        clear();
    }

}
