/* 
 * Copyright (C) ITC 2005 - International Institute for Geo-Information Science and Earth Observation
 * 
 * Author: Wim Koolhoven 
 * 
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by
 * the Free Software Foundation; either version 2.1 of the License, or (at
 * your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser
 * General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this library; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA 
 */
package org.itc.idv.math;


/**
 * Interface intended to describe and structure algorithms to be used for one pixel at a time.
 * The JAI calls these a {@code PointOpImage} 
 * <p>
 * First implementation as discussed on 17-08-05 by Martin Schouwenburg, Jan Hendrikse and Wim Koolhoven
 * 
 * @author Wim Koolhoven
 */
public interface Algorithm {
    /**
     * returns the Name of the Algorithm. 
     * This may not be NULL.
     */
    String getName();
    /**
     * may return the Description of the Algorithm.
     * Intended to give the end user more information. 
     */    
    String getDescription();
    /**
     * may return an URL to Documentation about the Algorithm.
     * Intended to show this documentation when the end user asks for help. 
     */
    String getDocURL();
    /**
     * may return the version of the Algorithm. 
     */
    String getVersion();
    /**
     * calculates the result of the Algorithm for these parameters.
     * Calling the Algorithm multiple times with the same params has to be consistent 
     * @param params the Array has to have as many elements as specified with {@link #getNrParams()}
     * @return the result
     * @throws RuntimeException 
     */
    double calculate(double[] params); 
    /**
     * returns the number of parameters 
     */
    int getNrParams();
    /**
     * returns the Name of the parameter
     * @param paramNr starts counting at 0
     * @return the Name of parameter {@code paramNr}
     * @throws IndexOutOfBoundsException when {@code paramNr} is negative or larger or equal to {#getNrParams()}
     */
    String getParamName(int paramNr); 
    /**
     * may return a Description of the parameter. Intended to give the end user more information.
     * @param paramNr starts counting at 0
     * @return the Name of parameter {@code paramNr}
     * @throws IndexOutOfBoundsException when {@code paramNr} is negative or larger or equal to {#getNrParams()}
     */
    String getParamDescription(int paramNr);
}
